<?php

namespace App\Http\Controllers\Admin;

use App\Helper\CommonHelper;
use App\Http\Controllers\Controller;
use App\Http\Requests\BlogStoreRequest;
use App\Http\Requests\BlogUpdateRequest;
use App\Models\Blog;
use Illuminate\Http\Request;
use Illuminate\View\View;

class BlogController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): View
    {
        $blogs = Blog::latest('created_at')->paginate(16);

        return view('pages.admin.blog.index', [
            'blogs' => $blogs,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(): View
    {
        return view('pages.admin.blog.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(BlogStoreRequest $request)
    {
        $uploadedImage = CommonHelper::storeImage($request->validated('image'), 'BLOG');
        $slug = CommonHelper::generateSlug($request->validated('slug'));

        Blog::create([
            'title' => $request->validated('title'),
            'image' => $uploadedImage,
            'slug' => $slug,
            'body' => $request->validated('body'),
            'summary' => $request->validated('summary'),
            'show' => $request->validated('show') === 'on',
            'user_id' => 1
        ]);

        return back();
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id): View
    {
        $blog = Blog::find($id);

        return view('pages.admin.blog.update', [
            'blog' => $blog
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(BlogUpdateRequest $request, string $id)
    {
        $blog = Blog::find($id);

        if ($request->has('image')) {
            $image = CommonHelper::updateImage($request->validated('image'), $blog->image, 'BLOG');
        } else {
            $image = $blog->image;
        }

        $slug = CommonHelper::generateSlug($request->validated('title'));

        $blog->update([
            'title' => $request->validated('title'),
            'image' => $image,
            'slug' => $slug,
            'body' => $request->validated('body'),
            'summary' => $request->validated('summary'),
            'show' => $request->validated('show') === 'on',
            'user_id' => 1
        ]);

        return redirect()->route('admin.blog');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $blog = Blog::find($id);

        if ($blog->exists()) {
            // Melakukan pengecekan apakah data image ada atau tidak null dan file image tersedia 
            if (!empty($blog->image)) {
                // Menghapus file image yang telah tersimpan agar tidak menumpuk, karen data akan dihapus
                CommonHelper::deleteImage($blog->image, 'BLOG');
            }

            $blog->delete();
        }

        return back();
    }
}
